# -*- encoding:utf8 -*-

# Fichier : jeuPendu.py
# Auteur : Jérôme Charrière
# Date : Mars 2019
# Version : 1.2
# Description :

# Module implémentant les différentes classes nécessaires à la réalisation
# du programme JeuPendu.

### Importation de fonctions nécessaires

from random import randint
from datetime import date


#### Classes nécessaires:

class Pendu(object):
    """Classe gérant le jeu du pendu"""

    PENDAISON = ['''

      +---+
      |   |
          |
          |
          |
          |
    =========''', '''

      +---+
      |   |
      O   |
          |
          |
          |
    =========''', '''

      +---+
      |   |
      O   |
      |   |
          |
          |
    =========''', '''

      +---+
      |   |
      O   |
     /|   |
          |
          |
    =========''', '''

      +---+
      |   |
      O   |
     /|\  |
          |
          |
    =========''', '''

      +---+
      |   |
      O   |
     /|\  |
     /    |
          |
    =========''', '''

      +---+
      |   |
      O   |
     /|\  |
     / \  |
          |
    =========''']                       # états du pendu

    mots = 'annexes/mots.txt'           # fichier contenant les mots à trouver


    def __init__(self, pseudo):
        """
        Constructeur du jeu enregistrant les informations du joueur,
        l'état du jeu et lançant le jeu
        """

        # Création de l'objet Joueur associé à la partie lancée
        self.joueur = Joueur(pseudo)
        # Etat du jeu
        self.jeuTermine = True
        # Lancement d'une partie
        self.start_it()


    def start_it(self):
        """
        start_it() --> None.
        Lance une nouvelle partie
        """

        # Initialisation des listes de lettres proposées (correctes et incorrectes)
        # sous forme d'attributs d'instance:
        self.lettresIncorrectes = []
        self.lettresCorrectes = []

        # Initialisation du mot secret sous forme d'attribut d'instance:
        self.motSecret = self.choisirMot()

        # Mise à jour de l'état du jeu
        self.jeuTermine = False

        while not self.jeuTermine:

            # lancement du jeu
            self.affichePendu()

            # proposer une lettre
            self.jouer()


    def choisirMot(self):
        """
        choisirMot() --> str.
        Retourne de manière aléatoire un mot situé dans le fichier <Pendu.mots>
        """

        fichier = open(Pendu.mots, 'r', encoding='utf8')

        listeMots = fichier.readlines()

        fichier.close()

        index = randint(0, len(listeMots)-1)

        mot = listeMots[index][:-1]     # suppression du caractère de fin de ligne

        return mot


    def affichePendu(self):
        """
        affichePendu() --> None.
        Affiche
        1) l'état du pendu situé dans la liste <Pendu.PENDAISON>
        à l'aide du nombre de lettres incorrectes déjà proposées situées dans la liste <lettresIncorrectes>
        2) l'énumération des lettres incorrectes déjà proposées situées dans la liste <lettresIncorrectes>
        3) l'état de la recherche du <motSecret> à partir de la liste <lettresCorrectes> des lettres de <motSecret>
        déjà trouvées.
        
        """
        # Afficher l'état du pendu:
        print(Pendu.PENDAISON[len(self.lettresIncorrectes)])
        print("\n")

        # Afficher la liste des lettres absentes du mot secret précédemment suggérées:
        print('Lettres absentes du mot secret:', end=' ')
        for lettre in self.lettresIncorrectes:
            print(lettre, end=' ')
        print("\n")
        
        # Afficher l'état du mot secret:
        etatMotSecret = '_' * len(self.motSecret)

        for i in range(len(self.motSecret)): # remplace les espaces (_) du mot secret par les lettres manquantes correctes
            if self.motSecret[i] in self.lettresCorrectes:
                etatMotSecret = etatMotSecret[:i] + self.motSecret[i] + etatMotSecret[i+1:]

        print("Etat du mot secret:", end=' ')
        for lettre in etatMotSecret: # montre l'état actuel du mot secret avec un espace entre chaque lettre
            print(lettre, end=' ')
        print("\n")


    def jouer(self):
        """
        jouer() --> None
        Demande à l'utilisateur d'entrer une lettre, controle sa validité, puis gère la lettre proposée
        """

        while 1:

            # Inviter l'utilisateur à entrer une lettre
            proposition = input("Proposez une lettre: ").lower()

            # Contrôler si la proposition est un seul caractère:
            if len(proposition) != 1:
                print('Entrez une seule lettre, s.v.p.')

            # Contrôler si la proposition a déjà été suggérée:
            elif proposition in self.lettresIncorrectes+self.lettresCorrectes:
                print('Vous avez déjà fait cette suggestion. Choisissez une autre lettre.')

            # Contrôler si la proposition est un caractère autorisé:
            elif proposition not in 'abcdefghijklmnopqrstuvwxyzâàéèêëîôûùç-':
                print('Entrez un caractère alphabétique, s.v.p.')

            # Garder la proposition:
            else:
                break

        # Si la proposition est située dans le motSecret,
        # 1) la rajouter à la liste des lettresCorrectes
        # 2) controler si le joueur a gagné, c'est-à-dire trouvé toutes les lettres
        if proposition in self.motSecret:
            self.lettresCorrectes.append(proposition)

            # Contrôler si le joueur à gagner:
            tousTrouve = True
            for i in range(len(self.motSecret)):
                if self.motSecret[i] not in self.lettresCorrectes:
                    tousTrouve = False
                    break
            if tousTrouve:
                print('Bravo! Le mot secret est "' + self.motSecret + '"! Vous avez gagné !')
                self.gestionPoints()
                self.jeuTermine = True
                self.end_it()


        # Si la proposition n'est pas située dans le motSecret,
        # 1) la rajouter à la liste des lettresIncorrectes
        # 2) controler si le joueur a perdu, c'est-à-dire si la pendaison est atteinte
        else:
            self.lettresIncorrectes.append(proposition)

            # Contrôler si le joueur a perdu:
            if len(self.lettresIncorrectes) == len(Pendu.PENDAISON) - 1:
                self.affichePendu()
                print('Vous avez perdu !\nVous aviez pourtant trouvé ' + str(len(self.lettresCorrectes)) + ' lettre(s) correcte(s)...\n\
    Le mot secret était: "' + self.motSecret + '"')
                self.jeuTermine = True
                self.end_it()

                
    def end_it(self):
        """
        end_it() --> None
        Gère la fin de partie
        """
        # Demander au joueur s'il veut rejouer:
        reponse = input("Voulez-vous jouer une nouvelle partie ? (oui ou non): ").lower()
        if reponse[0] == 'o':
            # Relancer une nouvelle partie
            self.start_it()
        else:
            # Termine le jeu
            # Afficher les meilleurs scores du joueur:
            print("\nVoici la liste de vos actuels meilleurs scores:\n")

            scores = self.joueur.trierScores()

            print("{:<3} \t {:^10} \t {:^10} \t {:>10}".format('Rang', 'Date', 'Mot', 'Points'))

            i = 0
            while i < 3 and i < len(scores):
                ligne = "{rang:<3} \t {date:^10} \t {mot:^10} \t {points:>10}".format(rang=i+1, date=scores[i][2], mot=scores[i][1], points=scores[i][0])
                print(ligne)
                i += 1               


    def gestionPoints(self):
        """
        gestionPoints() --> None
        Calcule les points générés par une partie et les transmet au joueur
        """

        totalEssais = len(self.lettresCorrectes + self.lettresIncorrectes)
        essaisGagnants = len(self.lettresCorrectes)
        points = (essaisGagnants*100)//totalEssais
        # Sauvegarde des points dans le fichier du joueur
        self.joueur.sauvegarder(self.motSecret, points)


      
class Joueur(object):
    """Classe gérant les joueurs du jeu du pendu"""

    def __init__(self, pseudo):
        """
        Constructeur d'un joueur enregistrant le pseudo et générant, si nécessaire,
        le fichier de sauvegarde des différentes parties gagnantes du joueur
        """
        # Pseudo du joueur
        self.pseudo = pseudo
        # Adresse du fichier contenant les résultats du joueur
        self.resultats = self.creerSauvegarde()

        
    def creerSauvegarde(self):
        """
        creerSauvegarde() --> str
        Crée, s'il n'existe pas, un fichier nommé 'pseudo.txt' dans le sous-dossier 'joueurs' du répertoire 'annexes'
        et y inscrit en première ligne le titre "Résultats de pseudo". Retourne l'adresse du fichier sous forme de string.
        """

        nomFichier = "annexes/joueurs/"+self.pseudo+'.txt'

        try:        # tente l'ouverture en mode lecture afin de savoir si le fichier existe
            fichier = open(nomFichier, 'r', encoding='utf8')
            
        except:     # crée le fichier s'il n'existe pas
            fichier = open(nomFichier, 'w', encoding='utf8')
            fichier.write("Résultats de "+ self.pseudo + ":\n")
            fichier.close()
            
        else:
            fichier.close()

        return nomFichier

        
    def sauvegarder(self, motCherche, points):
        """
        sauvegarder(str motCherche, int points) --> None
        Enregistre sur une seule ligne du fichier associé au joueur le nombre de <points>
        réalisé lors de la recherche du <motCherche> au format suivant:

        date(31/3/2019):motCherche@points
        """

        fichier = open(self.resultats, 'a', encoding='utf8')

        today = date.today()        # récupération de la date d'aujourd'hui

        ligne = '{date}:{mot}@{pts}\n'.\
                format(date=str(today.day)+'/'+str(today.month)+'/'+str(today.year),\
                       mot = motCherche,\
                       pts = points)

        fichier.write(ligne)

        fichier.close()

    def trierScores(self):
        """
        trierScores() --> list
        Récupère tous les scores situés dans chacune des lignes du fichier associé au
        joueur et retourne la liste de ces scores sous forme de tuples (points, mot, date)
        triée en fonction du nombre de points
        """
 
        fichier = open(self.resultats, 'r', encoding='utf8')

        resultats = []

        fichier.readline()                                      # la première ligne contient le titre

        score = fichier.readline()

        while score != "":
            date = score.split(":")[0]                          # récupère la date
            mot = (score.split(":")[1]).split("@")[0]           # récupère le mot
            points = (score.split(":")[1]).split("@")[1][:-1]   # récupère les points associés
            
            resultats.append((int(points), mot, date))               # ajoute le tuple (points, mot, date) à la liste
            
            score = fichier.readline()

        fichier.close()

        resultats.sort()
        resultats.reverse()                                     # trie la liste de manière décroissante

        return resultats
